package one.netspace.sdk;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class NetspaceSDK {
    private final String apiKey;
    private final String secretKey;
    private final String baseUrl;
    private final int timeout;
    private final Gson gson;

    public NetspaceSDK(String apiKey, String secretKey) {
        this(apiKey, secretKey, "production", null, 30000);
    }

    public NetspaceSDK(String apiKey, String secretKey, String environment) {
        this(apiKey, secretKey, environment, null, 30000);
    }

    public NetspaceSDK(String apiKey, String secretKey, String environment, String baseUrl, int timeout) {
        if (apiKey == null || apiKey.isEmpty() || secretKey == null || secretKey.isEmpty()) {
            throw new NetspaceException("API Key and Secret Key are required");
        }

        this.apiKey = apiKey;
        this.secretKey = secretKey;
        this.timeout = timeout;
        this.gson = new Gson();

        if (baseUrl != null && !baseUrl.isEmpty()) {
            this.baseUrl = baseUrl;
        } else if ("sandbox".equals(environment)) {
            this.baseUrl = "https://sandbox-api.netspace.one";
        } else {
            this.baseUrl = "https://api.netspace.one";
        }
    }

    public LaunchResponse launchGame(LaunchGameParams params) {
        Map<String, Object> body = new HashMap<>();
        body.put("session_id", params.getSessionId());
        body.put("player_id", params.getPlayerId());
        body.put("game_id", params.getGameId());
        body.put("currency", params.getCurrency());
        body.put("language", params.getLanguage() != null ? params.getLanguage() : "en");
        body.put("return_url", params.getReturnUrl());
        body.put("client_ip", params.getClientIp());
        body.put("metadata", params.getMetadata());

        String response = request("POST", "/game/launch", body);
        return gson.fromJson(response, LaunchResponse.class);
    }

    public LaunchResponse launchDemo(LaunchDemoParams params) {
        Map<String, Object> body = new HashMap<>();
        body.put("game_id", params.getGameId());
        body.put("currency", params.getCurrency() != null ? params.getCurrency() : "USD");
        body.put("language", params.getLanguage() != null ? params.getLanguage() : "en");
        body.put("return_url", params.getReturnUrl());
        body.put("demo_balance", params.getDemoBalance() > 0 ? params.getDemoBalance() : 10000);

        String response = request("POST", "/game/launch-demo", body);
        return gson.fromJson(response, LaunchResponse.class);
    }

    public CurrenciesResponse getCurrencies() {
        String response = request("GET", "/game/currencies", null);
        return gson.fromJson(response, CurrenciesResponse.class);
    }

    public List<GameInfo> getGames() {
        String response = request("GET", "/game/games", null);
        return gson.fromJson(response, new TypeToken<List<GameInfo>>(){}.getType());
    }

    private String request(String method, String path, Map<String, Object> body) {
        try {
            URL url = new URL(baseUrl + path);
            HttpURLConnection conn = (HttpURLConnection) url.openConnection();
            conn.setRequestMethod(method);
            conn.setConnectTimeout(timeout);
            conn.setReadTimeout(timeout);

            String timestamp = String.valueOf(System.currentTimeMillis());
            String bodyJson = body != null ? gson.toJson(body) : "";
            String signature = sign(timestamp, bodyJson);

            conn.setRequestProperty("Content-Type", "application/json");
            conn.setRequestProperty("X-API-Key", apiKey);
            conn.setRequestProperty("X-Timestamp", timestamp);
            conn.setRequestProperty("X-Signature", signature);

            if ("POST".equals(method) && body != null) {
                conn.setDoOutput(true);
                try (OutputStream os = conn.getOutputStream()) {
                    os.write(bodyJson.getBytes(StandardCharsets.UTF_8));
                }
            }

            int responseCode = conn.getResponseCode();
            InputStream is = responseCode >= 400 ? conn.getErrorStream() : conn.getInputStream();

            StringBuilder response = new StringBuilder();
            try (BufferedReader br = new BufferedReader(new InputStreamReader(is, StandardCharsets.UTF_8))) {
                String line;
                while ((line = br.readLine()) != null) {
                    response.append(line);
                }
            }

            if (responseCode >= 400) {
                Map<String, Object> error = gson.fromJson(response.toString(), new TypeToken<Map<String, Object>>(){}.getType());
                String message = error.get("message") != null ? error.get("message").toString() : "Unknown error";
                String code = error.get("code") != null ? error.get("code").toString() : "UNKNOWN_ERROR";
                throw new NetspaceException(message, responseCode, code);
            }

            return response.toString();

        } catch (IOException e) {
            throw new NetspaceException("HTTP request failed: " + e.getMessage());
        }
    }

    private String sign(String timestamp, String body) {
        try {
            String stringToSign = timestamp + body;
            Mac mac = Mac.getInstance("HmacSHA256");
            SecretKeySpec secretKeySpec = new SecretKeySpec(secretKey.getBytes(StandardCharsets.UTF_8), "HmacSHA256");
            mac.init(secretKeySpec);
            byte[] hash = mac.doFinal(stringToSign.getBytes(StandardCharsets.UTF_8));

            StringBuilder hexString = new StringBuilder();
            for (byte b : hash) {
                String hex = Integer.toHexString(0xff & b);
                if (hex.length() == 1) hexString.append('0');
                hexString.append(hex);
            }
            return hexString.toString();

        } catch (Exception e) {
            throw new NetspaceException("Failed to generate signature: " + e.getMessage());
        }
    }
}
