# Netspace ONE Java SDK

Java SDK for integrating Netspace ONE games into your platform.

## Requirements

- Java 11 or higher
- Gson library

## Installation

### Maven

```xml
<dependency>
    <groupId>one.netspace</groupId>
    <artifactId>netspace-sdk</artifactId>
    <version>1.0.0</version>
</dependency>
```

### Gradle

```groovy
implementation 'one.netspace:netspace-sdk:1.0.0'
```

## Quick Start

```java
import one.netspace.sdk.*;

public class Example {
    public static void main(String[] args) {
        // Initialize SDK
        NetspaceSDK netspace = new NetspaceSDK(
            "your-api-key",
            "your-secret-key",
            "production"  // or "sandbox"
        );

        try {
            // Launch a game
            LaunchResponse session = netspace.launchGame(
                new LaunchGameParams()
                    .setSessionId("your-session-id")
                    .setPlayerId("player-123")
                    .setGameId("lotterix")
                    .setCurrency("USD")
                    .setLanguage("en")
                    .setReturnUrl("https://your-site.com/lobby")
                    .setClientIp("192.168.1.1")
            );

            // Redirect player to game
            String gameUrl = session.getGameUrl();
            System.out.println("Game URL: " + gameUrl);

        } catch (NetspaceException e) {
            System.err.println("Error: " + e.getMessage());
            System.err.println("Code: " + e.getErrorCode());
        }
    }
}
```

## API Reference

### Constructor

```java
// Simple
NetspaceSDK netspace = new NetspaceSDK("api-key", "secret-key");

// With environment
NetspaceSDK netspace = new NetspaceSDK("api-key", "secret-key", "sandbox");

// Full options
NetspaceSDK netspace = new NetspaceSDK(
    "api-key",
    "secret-key",
    "production",
    "https://custom-api.example.com",  // custom base URL
    30000  // timeout in milliseconds
);
```

### Methods

#### `launchGame(LaunchGameParams params)`

Launch a real-money game session.

```java
LaunchResponse session = netspace.launchGame(
    new LaunchGameParams()
        .setSessionId("unique-session-id")  // Required
        .setPlayerId("player-123")          // Required
        .setGameId("lotterix")              // Required
        .setCurrency("USD")                 // Required
        .setClientIp("192.168.1.1")         // Required
        .setLanguage("en")                  // Optional
        .setReturnUrl("https://...")        // Optional
        .setMetadata(Map.of("custom", "data"))  // Optional
);
```

**Response:**
```java
session.getOperatorSessionId();  // Your session ID
session.getNetspaceSessionId();  // Netspace internal ID
session.getSessionToken();       // JWT token
session.getGameUrl();            // URL to load the game
session.getExpiresAt();          // Expiration timestamp
```

#### `launchDemo(LaunchDemoParams params)`

Launch a demo game session.

```java
LaunchResponse session = netspace.launchDemo(
    new LaunchDemoParams()
        .setGameId("lotterix")      // Required
        .setCurrency("USD")         // Optional
        .setLanguage("en")          // Optional
        .setDemoBalance(10000)      // Optional: balance in cents
);
```

#### `getCurrencies()`

Get list of supported currencies.

```java
CurrenciesResponse currencies = netspace.getCurrencies();
currencies.getFiat().getCurrencies();   // List of fiat currencies
currencies.getCrypto().getCurrencies(); // List of crypto currencies
```

#### `getGames()`

Get available games catalog.

```java
List<GameInfo> games = netspace.getGames();
for (GameInfo game : games) {
    System.out.println(game.getId() + ": " + game.getName());
}
```

## Error Handling

```java
try {
    LaunchResponse session = netspace.launchGame(...);
} catch (NetspaceException e) {
    String message = e.getMessage();        // Error message
    int statusCode = e.getStatusCode();     // HTTP status code
    String errorCode = e.getErrorCode();    // Error code
}
```

## Spring Boot Integration

```java
// application.properties
netspace.api-key=${NETSPACE_API_KEY}
netspace.secret-key=${NETSPACE_SECRET_KEY}
netspace.environment=production

// NetspaceConfig.java
@Configuration
public class NetspaceConfig {
    @Value("${netspace.api-key}")
    private String apiKey;

    @Value("${netspace.secret-key}")
    private String secretKey;

    @Value("${netspace.environment}")
    private String environment;

    @Bean
    public NetspaceSDK netspaceSDK() {
        return new NetspaceSDK(apiKey, secretKey, environment);
    }
}

// GameController.java
@RestController
public class GameController {
    @Autowired
    private NetspaceSDK netspace;

    @GetMapping("/launch/{gameId}")
    public ResponseEntity<Void> launchGame(
        @PathVariable String gameId,
        HttpServletRequest request,
        @AuthenticationPrincipal User user
    ) {
        LaunchResponse session = netspace.launchGame(
            new LaunchGameParams()
                .setSessionId(request.getSession().getId())
                .setPlayerId(user.getId().toString())
                .setGameId(gameId)
                .setCurrency("USD")
                .setClientIp(request.getRemoteAddr())
        );

        return ResponseEntity
            .status(HttpStatus.FOUND)
            .header("Location", session.getGameUrl())
            .build();
    }
}
```

## Android Integration

```java
// In your Activity or ViewModel
public class GameActivity extends AppCompatActivity {
    private NetspaceSDK netspace;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        netspace = new NetspaceSDK(
            BuildConfig.NETSPACE_API_KEY,
            BuildConfig.NETSPACE_SECRET_KEY
        );
    }

    public void launchGame(String gameId) {
        new Thread(() -> {
            try {
                LaunchResponse session = netspace.launchGame(
                    new LaunchGameParams()
                        .setSessionId(UUID.randomUUID().toString())
                        .setPlayerId(getCurrentUserId())
                        .setGameId(gameId)
                        .setCurrency("USD")
                        .setClientIp(getDeviceIp())
                );

                runOnUiThread(() -> {
                    Intent intent = new Intent(Intent.ACTION_VIEW);
                    intent.setData(Uri.parse(session.getGameUrl()));
                    startActivity(intent);
                });

            } catch (NetspaceException e) {
                runOnUiThread(() -> 
                    Toast.makeText(this, e.getMessage(), Toast.LENGTH_LONG).show()
                );
            }
        }).start();
    }
}
```

## Support

- Documentation: https://netspace.one/api
- Email: tech@netspace.one

## License

MIT
