<?php

namespace Netspace;

class NetspaceSDK
{
    private string $apiKey;
    private string $secretKey;
    private string $baseUrl;
    private int $timeout;

    public function __construct(array $config)
    {
        $this->apiKey = $config['apiKey'] ?? '';
        $this->secretKey = $config['secretKey'] ?? '';
        $this->baseUrl = $config['baseUrl'] ?? $this->getDefaultBaseUrl($config['environment'] ?? 'production');
        $this->timeout = $config['timeout'] ?? 30;

        if (empty($this->apiKey) || empty($this->secretKey)) {
            throw new NetspaceException('API Key and Secret Key are required');
        }
    }

    private function getDefaultBaseUrl(string $environment): string
    {
        return $environment === 'sandbox'
            ? 'https://sandbox-api.netspace.one'
            : 'https://api.netspace.one';
    }

    public function launchGame(array $params): array
    {
        $this->validateRequired($params, ['sessionId', 'playerId', 'gameId', 'currency', 'clientIp']);

        return $this->request('POST', '/game/launch', [
            'session_id' => $params['sessionId'],
            'player_id' => $params['playerId'],
            'game_id' => $params['gameId'],
            'currency' => $params['currency'],
            'language' => $params['language'] ?? 'en',
            'return_url' => $params['returnUrl'] ?? null,
            'client_ip' => $params['clientIp'],
            'metadata' => $params['metadata'] ?? null,
        ]);
    }

    public function launchDemo(array $params): array
    {
        $this->validateRequired($params, ['gameId']);

        return $this->request('POST', '/game/launch-demo', [
            'game_id' => $params['gameId'],
            'currency' => $params['currency'] ?? 'USD',
            'language' => $params['language'] ?? 'en',
            'return_url' => $params['returnUrl'] ?? null,
            'demo_balance' => $params['demoBalance'] ?? 10000,
        ]);
    }

    public function getCurrencies(): array
    {
        return $this->request('GET', '/game/currencies');
    }

    public function getGames(): array
    {
        return $this->request('GET', '/game/games');
    }

    private function request(string $method, string $path, ?array $body = null): array
    {
        $url = $this->baseUrl . $path;
        $timestamp = (string) round(microtime(true) * 1000);
        $bodyJson = $body ? json_encode($body) : '';

        $signature = $this->sign($timestamp, $bodyJson);

        $headers = [
            'Content-Type: application/json',
            'X-API-Key: ' . $this->apiKey,
            'X-Timestamp: ' . $timestamp,
            'X-Signature: ' . $signature,
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, $this->timeout);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $bodyJson);
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            throw new NetspaceException('HTTP request failed: ' . $error);
        }

        $data = json_decode($response, true);

        if ($httpCode >= 400) {
            $message = $data['message'] ?? 'Unknown error';
            $code = $data['code'] ?? 'UNKNOWN_ERROR';
            throw new NetspaceException($message, $httpCode, $code);
        }

        return $data;
    }

    private function sign(string $timestamp, string $body): string
    {
        $stringToSign = $timestamp . $body;
        return hash_hmac('sha256', $stringToSign, $this->secretKey);
    }

    private function validateRequired(array $params, array $required): void
    {
        foreach ($required as $field) {
            if (empty($params[$field])) {
                throw new NetspaceException("Missing required parameter: {$field}");
            }
        }
    }
}
