# Netspace ONE SDK

JavaScript/TypeScript SDK for integrating Netspace ONE games into your platform.

## Installation

### NPM
```bash
npm install @netspace.one/sdk
```

### CDN (Browser)
```html
<script src="https://cdn.netspace.one/sdk/netspace-sdk.js"></script>
```

> **Note:** The SDK is hosted on our CDN at `cdn.netspace.one`

## Quick Start

### ES Modules / TypeScript
```typescript
import { NetspaceSDK } from '@netspace.one/sdk';

const netspace = new NetspaceSDK({
  apiKey: 'your-api-key',
  secretKey: 'your-secret-key',
  environment: 'production' // or 'sandbox'
});

// Launch a game
const session = await netspace.launchGame({
  sessionId: 'your-session-id',
  playerId: 'player-123',
  gameId: 'lotterix',
  currency: 'USD',
  language: 'en',
  returnUrl: 'https://your-site.com/lobby',
  clientIp: '192.168.1.1'
});

// Open in iframe
netspace.openInIframe('game-container', session.game_url);
```

### Browser (UMD)
```html
<div id="game-container" style="width: 100%; height: 600px;"></div>

<script src="https://cdn.netspace.one/sdk/netspace-sdk.js"></script>
<script>
  const netspace = new NetspaceSDK.NetspaceSDK({
    apiKey: 'your-api-key',
    secretKey: 'your-secret-key'
  });

  async function launchGame() {
    const session = await netspace.launchGame({
      sessionId: 'session-' + Date.now(),
      playerId: 'player-123',
      gameId: 'lotterix',
      currency: 'USD',
      language: 'en',
      returnUrl: window.location.href,
      clientIp: '0.0.0.0' // Get from your backend
    });

    netspace.openInIframe('game-container', session.game_url);
  }

  launchGame();
</script>
```

## API Reference

### Constructor

```typescript
new NetspaceSDK(config: NetspaceConfig)
```

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `apiKey` | string | Yes | Your operator API key |
| `secretKey` | string | Yes | Your operator secret key |
| `environment` | 'production' \| 'sandbox' | No | API environment (default: 'production') |
| `baseUrl` | string | No | Custom API base URL |

### Methods

#### `launchGame(params): Promise<LaunchResponse>`

Launch a real-money game session.

```typescript
const session = await netspace.launchGame({
  sessionId: 'unique-session-id',    // Your internal session ID
  playerId: 'player-123',            // Your internal player ID
  gameId: 'lotterix',                // Game identifier
  currency: 'USD',                   // Player's currency
  language: 'en',                    // UI language (ISO 639-1)
  returnUrl: 'https://...',          // URL to redirect on exit
  clientIp: '192.168.1.1',           // Player's IP address
  metadata: { custom: 'data' }       // Optional metadata
});
```

**Response:**
```typescript
{
  operator_session_id: string;   // Your session ID
  netspace_session_id: string;   // Netspace internal session ID
  session_token: string;         // JWT token for the session
  game_url: string;              // URL to load the game
  expires_at: string;            // Session expiration (ISO 8601)
}
```

#### `launchDemo(params): Promise<LaunchResponse>`

Launch a demo game session (no real money).

```typescript
const session = await netspace.launchDemo({
  gameId: 'lotterix',
  currency: 'USD',           // Optional, default: USD
  language: 'en',            // Optional, default: en
  demoBalance: 10000         // Optional, demo balance in cents
});
```

#### `getCurrencies(): Promise<CurrenciesResponse>`

Get list of supported currencies.

```typescript
const currencies = await netspace.getCurrencies();
// { fiat: { count: 18, currencies: [...] }, crypto: { count: 15, currencies: [...] } }
```

#### `getGames(): Promise<GameInfo[]>`

Get available games catalog.

```typescript
const games = await netspace.getGames();
// [{ id: 'lotterix', name: 'Lotterix', rtp: 72, ... }]
```

#### `openInIframe(containerId, gameUrl, options?)`

Open game in an iframe element.

```typescript
netspace.openInIframe('game-container', session.game_url, {
  width: '100%',
  height: '600px',
  allowFullscreen: true
});
```

#### `openInWindow(gameUrl, options?)`

Open game in a new browser window.

```typescript
netspace.openInWindow(session.game_url, {
  width: 1280,
  height: 720,
  name: 'netspace-game'
});
```

#### `closeGame()`

Close the game iframe.

```typescript
netspace.closeGame();
```

#### `on(event, callback)`

Subscribe to game events.

```typescript
netspace.on('exit', (event) => {
  console.log('Player exited the game');
  // Redirect to lobby or show message
});

netspace.on('error', (event) => {
  console.error('Game error:', event.data);
});

netspace.on('session_expired', (event) => {
  console.log('Session expired, redirecting...');
});
```

**Available events:**
- `exit` - Player clicked exit button
- `error` - Game error occurred
- `session_expired` - Session token expired
- `balance_low` - Player balance is low
- `bet` - Player placed a bet
- `win` - Player won

#### `off(event, callback?)`

Unsubscribe from game events.

```typescript
netspace.off('exit'); // Remove all listeners
netspace.off('exit', myCallback); // Remove specific listener
```

#### `destroy()`

Cleanup SDK instance.

```typescript
netspace.destroy();
```

## Supported Currencies

### Fiat (18)
USD, EUR, GBP, BRL, ARS, MXN, CLP, COP, PEN, CAD, AUD, JPY, CNY, INR, KRW, TRY, ZAR, NGN

### Crypto (15)
BTC, ETH, USDT, USDC, BNB, XRP, SOL, ADA, DOGE, TRX, TON, MATIC, LTC, AVAX, DOT

## Supported Languages

ES, EN, PT, AR, DE, FR, HI, ID, IT, JA, KO, RU, TR, ZH

## Error Handling

```typescript
import { NetspaceSDK, NetspaceError } from '@netspace.one/sdk';

try {
  const session = await netspace.launchGame({ ... });
} catch (error) {
  if (error instanceof NetspaceError) {
    console.error(`Error ${error.statusCode}: ${error.message}`);
    console.error(`Code: ${error.code}`);
  }
}
```

## Security

### Request Signing

All authenticated requests are signed using HMAC-SHA256:

```
signature = HMAC-SHA256(secretKey, timestamp + requestBody)
```

Headers sent:
- `x-api-key`: Your API key
- `x-timestamp`: Unix timestamp in milliseconds
- `x-signature`: HMAC-SHA256 signature

### Best Practices

1. **Never expose your secret key** in client-side code
2. **Use a backend proxy** for production to sign requests server-side
3. **Validate returnUrl** to prevent open redirects
4. **Use HTTPS** for all communications

## TypeScript Support

Full TypeScript definitions are included:

```typescript
import { 
  NetspaceSDK, 
  NetspaceConfig,
  LaunchGameParams,
  LaunchResponse,
  GameEventData,
  NetspaceError 
} from '@netspace.one/sdk';
```

## Browser Support

- Chrome 80+
- Firefox 75+
- Safari 13+
- Edge 80+

## Support

- Documentation: https://docs.netspace.one
- Email: support@netspace.one
- API Status: https://status.netspace.one

## License

MIT
