/**
 * Netspace ONE SDK
 *
 * JavaScript SDK for game operators to integrate Netspace ONE games.
 *
 * @example
 * ```javascript
 * const netspace = new NetspaceSDK({
 *   apiKey: 'your-api-key',
 *   secretKey: 'your-secret-key',
 *   environment: 'production'
 * });
 *
 * // Launch a game
 * const session = await netspace.launchGame({
 *   sessionId: 'operator-session-123',
 *   playerId: 'player-456',
 *   gameId: 'lotterix',
 *   currency: 'USD',
 *   language: 'en',
 *   returnUrl: 'https://your-site.com/lobby',
 *   clientIp: '192.168.1.1'
 * });
 *
 * // Open game in iframe
 * netspace.openInIframe('game-container', session.game_url);
 * ```
 */
export interface NetspaceConfig {
    apiKey: string;
    secretKey: string;
    environment?: 'production' | 'sandbox';
    baseUrl?: string;
}
export interface LaunchGameParams {
    sessionId: string;
    playerId: string;
    gameId: string;
    currency: string;
    language: string;
    returnUrl: string;
    clientIp: string;
    metadata?: Record<string, unknown>;
}
export interface LaunchDemoParams {
    gameId: string;
    playerId?: string;
    currency?: string;
    language?: string;
    returnUrl?: string;
    clientIp?: string;
    demoBalance?: number;
}
export interface LaunchResponse {
    operator_session_id: string;
    netspace_session_id: string;
    session_token: string;
    game_url: string;
    expires_at: string;
}
export interface GameInfo {
    id: string;
    name: string;
    slug: string;
    provider: string;
    category: string;
    rtp: number;
    volatility: string;
    thumbnail: string;
    isActive: boolean;
}
export interface CurrencyInfo {
    code: string;
    name: string;
    symbol: string;
    decimals: number;
}
export interface CurrenciesResponse {
    fiat: {
        count: number;
        currencies: CurrencyInfo[];
    };
    crypto: {
        count: number;
        currencies: CurrencyInfo[];
    };
    total: number;
}
export interface GameEventData {
    type: 'exit' | 'error' | 'session_expired' | 'balance_low' | 'bet' | 'win';
    data?: Record<string, unknown>;
}
type EventCallback = (event: GameEventData) => void;
export declare class NetspaceSDK {
    private config;
    private eventListeners;
    private gameIframe;
    constructor(config: NetspaceConfig);
    /**
     * Generate HMAC-SHA256 signature for API requests
     */
    private generateSignature;
    /**
     * Make authenticated API request
     */
    private request;
    /**
     * Launch a game session (real money)
     */
    launchGame(params: LaunchGameParams): Promise<LaunchResponse>;
    /**
     * Launch a demo game session (no real money)
     */
    launchDemo(params: LaunchDemoParams): Promise<LaunchResponse>;
    /**
     * Get list of supported currencies
     */
    getCurrencies(): Promise<CurrenciesResponse>;
    /**
     * Get game catalog
     */
    getGames(): Promise<GameInfo[]>;
    /**
     * Open game in an iframe
     */
    openInIframe(containerId: string, gameUrl: string, options?: {
        width?: string;
        height?: string;
        allowFullscreen?: boolean;
    }): HTMLIFrameElement;
    /**
     * Open game in a new window/tab
     */
    openInWindow(gameUrl: string, options?: {
        width?: number;
        height?: number;
        name?: string;
    }): Window | null;
    /**
     * Close the game iframe
     */
    closeGame(): void;
    /**
     * Subscribe to game events
     */
    on(event: string, callback: EventCallback): void;
    /**
     * Unsubscribe from game events
     */
    off(event: string, callback?: EventCallback): void;
    /**
     * Handle postMessage events from game iframe
     */
    private handleGameMessage;
    /**
     * Destroy SDK instance and cleanup
     */
    destroy(): void;
}
/**
 * Custom error class for Netspace SDK
 */
export declare class NetspaceError extends Error {
    statusCode: number;
    code?: string;
    constructor(message: string, statusCode: number, code?: string);
}
export default NetspaceSDK;
