(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.NetspaceSDK = {}));
})(this, (function (exports) { 'use strict';

    /**
     * Netspace ONE SDK
     *
     * JavaScript SDK for game operators to integrate Netspace ONE games.
     *
     * @example
     * ```javascript
     * const netspace = new NetspaceSDK({
     *   apiKey: 'your-api-key',
     *   secretKey: 'your-secret-key',
     *   environment: 'production'
     * });
     *
     * // Launch a game
     * const session = await netspace.launchGame({
     *   sessionId: 'operator-session-123',
     *   playerId: 'player-456',
     *   gameId: 'lotterix',
     *   currency: 'USD',
     *   language: 'en',
     *   returnUrl: 'https://your-site.com/lobby',
     *   clientIp: '192.168.1.1'
     * });
     *
     * // Open game in iframe
     * netspace.openInIframe('game-container', session.game_url);
     * ```
     */
    const DEFAULT_URLS = {
        production: 'https://api.netspace.one',
        sandbox: 'https://sandbox-api.netspace.one',
    };
    class NetspaceSDK {
        constructor(config) {
            this.eventListeners = new Map();
            this.gameIframe = null;
            const environment = config.environment || 'production';
            this.config = {
                apiKey: config.apiKey,
                secretKey: config.secretKey,
                environment,
                baseUrl: config.baseUrl || DEFAULT_URLS[environment],
            };
            // Listen for postMessage events from game iframe
            if (typeof window !== 'undefined') {
                window.addEventListener('message', this.handleGameMessage.bind(this));
            }
        }
        /**
         * Generate HMAC-SHA256 signature for API requests
         */
        async generateSignature(timestamp, body) {
            const message = `${timestamp}${body}`;
            if (typeof window !== 'undefined' && window.crypto?.subtle) {
                // Browser environment
                const encoder = new TextEncoder();
                const keyData = encoder.encode(this.config.secretKey);
                const messageData = encoder.encode(message);
                const key = await window.crypto.subtle.importKey('raw', keyData, { name: 'HMAC', hash: 'SHA-256' }, false, ['sign']);
                const signature = await window.crypto.subtle.sign('HMAC', key, messageData);
                return Array.from(new Uint8Array(signature))
                    .map(b => b.toString(16).padStart(2, '0'))
                    .join('');
            }
            else {
                // Node.js environment
                const crypto = await import('crypto');
                return crypto.createHmac('sha256', this.config.secretKey)
                    .update(message)
                    .digest('hex');
            }
        }
        /**
         * Make authenticated API request
         */
        async request(method, endpoint, body, authenticated = true) {
            const url = `${this.config.baseUrl}${endpoint}`;
            const timestamp = Date.now().toString();
            const bodyString = body ? JSON.stringify(body) : '';
            const headers = {
                'Content-Type': 'application/json',
            };
            if (authenticated) {
                headers['x-api-key'] = this.config.apiKey;
                headers['x-timestamp'] = timestamp;
                headers['x-signature'] = await this.generateSignature(timestamp, bodyString);
            }
            const response = await fetch(url, {
                method,
                headers,
                body: method === 'POST' ? bodyString : undefined,
            });
            if (!response.ok) {
                const error = await response.json().catch(() => ({ message: 'Unknown error' }));
                throw new NetspaceError(error.message || `HTTP ${response.status}`, response.status, error.code);
            }
            return response.json();
        }
        /**
         * Launch a game session (real money)
         */
        async launchGame(params) {
            return this.request('POST', '/game/launch', {
                session_id: params.sessionId,
                player_id: params.playerId,
                game_id: params.gameId,
                currency: params.currency,
                language: params.language,
                return_url: params.returnUrl,
                client_ip: params.clientIp,
                metadata: params.metadata,
            });
        }
        /**
         * Launch a demo game session (no real money)
         */
        async launchDemo(params) {
            return this.request('POST', '/game/launch-demo', {
                game_id: params.gameId,
                player_id: params.playerId,
                currency: params.currency,
                language: params.language,
                return_url: params.returnUrl,
                client_ip: params.clientIp,
                demo_balance: params.demoBalance,
            }, false);
        }
        /**
         * Get list of supported currencies
         */
        async getCurrencies() {
            return this.request('GET', '/game/currencies', undefined, false);
        }
        /**
         * Get game catalog
         */
        async getGames() {
            return this.request('GET', '/game/catalog', undefined, false);
        }
        /**
         * Open game in an iframe
         */
        openInIframe(containerId, gameUrl, options) {
            const container = document.getElementById(containerId);
            if (!container) {
                throw new NetspaceError(`Container element '${containerId}' not found`, 404);
            }
            // Remove existing iframe if any
            if (this.gameIframe) {
                this.gameIframe.remove();
            }
            const iframe = document.createElement('iframe');
            iframe.src = gameUrl;
            iframe.style.width = options?.width || '100%';
            iframe.style.height = options?.height || '100%';
            iframe.style.border = 'none';
            iframe.allow = 'autoplay; fullscreen; payment';
            if (options?.allowFullscreen !== false) {
                iframe.allowFullscreen = true;
            }
            container.appendChild(iframe);
            this.gameIframe = iframe;
            return iframe;
        }
        /**
         * Open game in a new window/tab
         */
        openInWindow(gameUrl, options) {
            const width = options?.width || 1280;
            const height = options?.height || 720;
            const name = options?.name || 'netspace-game';
            const left = (screen.width - width) / 2;
            const top = (screen.height - height) / 2;
            return window.open(gameUrl, name, `width=${width},height=${height},left=${left},top=${top},resizable=yes,scrollbars=no`);
        }
        /**
         * Close the game iframe
         */
        closeGame() {
            if (this.gameIframe) {
                this.gameIframe.remove();
                this.gameIframe = null;
            }
        }
        /**
         * Subscribe to game events
         */
        on(event, callback) {
            const listeners = this.eventListeners.get(event) || [];
            listeners.push(callback);
            this.eventListeners.set(event, listeners);
        }
        /**
         * Unsubscribe from game events
         */
        off(event, callback) {
            if (!callback) {
                this.eventListeners.delete(event);
            }
            else {
                const listeners = this.eventListeners.get(event) || [];
                this.eventListeners.set(event, listeners.filter(cb => cb !== callback));
            }
        }
        /**
         * Handle postMessage events from game iframe
         */
        handleGameMessage(event) {
            // Validate origin
            if (!event.origin.includes('netspace.one')) {
                return;
            }
            const data = event.data;
            if (!data?.type) {
                return;
            }
            // Emit to listeners
            const listeners = this.eventListeners.get(data.type) || [];
            listeners.forEach(callback => callback(data));
            // Also emit to 'all' listeners
            const allListeners = this.eventListeners.get('all') || [];
            allListeners.forEach(callback => callback(data));
        }
        /**
         * Destroy SDK instance and cleanup
         */
        destroy() {
            this.closeGame();
            this.eventListeners.clear();
            if (typeof window !== 'undefined') {
                window.removeEventListener('message', this.handleGameMessage.bind(this));
            }
        }
    }
    /**
     * Custom error class for Netspace SDK
     */
    class NetspaceError extends Error {
        constructor(message, statusCode, code) {
            super(message);
            this.name = 'NetspaceError';
            this.statusCode = statusCode;
            this.code = code;
        }
    }

    exports.NetspaceError = NetspaceError;
    exports.NetspaceSDK = NetspaceSDK;
    exports.default = NetspaceSDK;

    Object.defineProperty(exports, '__esModule', { value: true });

}));
