/**
 * Netspace ONE SDK
 * 
 * JavaScript SDK for game operators to integrate Netspace ONE games.
 * 
 * @example
 * ```javascript
 * const netspace = new NetspaceSDK({
 *   apiKey: 'your-api-key',
 *   secretKey: 'your-secret-key',
 *   environment: 'production'
 * });
 * 
 * // Launch a game
 * const session = await netspace.launchGame({
 *   sessionId: 'operator-session-123',
 *   playerId: 'player-456',
 *   gameId: 'lotterix',
 *   currency: 'USD',
 *   language: 'en',
 *   returnUrl: 'https://your-site.com/lobby',
 *   clientIp: '192.168.1.1'
 * });
 * 
 * // Open game in iframe
 * netspace.openInIframe('game-container', session.game_url);
 * ```
 */

export interface NetspaceConfig {
  apiKey: string;
  secretKey: string;
  environment?: 'production' | 'sandbox';
  baseUrl?: string;
}

export interface LaunchGameParams {
  sessionId: string;
  playerId: string;
  gameId: string;
  currency: string;
  language: string;
  returnUrl: string;
  clientIp: string;
  metadata?: Record<string, unknown>;
}

export interface LaunchDemoParams {
  gameId: string;
  playerId?: string;
  currency?: string;
  language?: string;
  returnUrl?: string;
  clientIp?: string;
  demoBalance?: number;
}

export interface LaunchResponse {
  operator_session_id: string;
  netspace_session_id: string;
  session_token: string;
  game_url: string;
  expires_at: string;
}

export interface GameInfo {
  id: string;
  name: string;
  slug: string;
  provider: string;
  category: string;
  rtp: number;
  volatility: string;
  thumbnail: string;
  isActive: boolean;
}

export interface CurrencyInfo {
  code: string;
  name: string;
  symbol: string;
  decimals: number;
}

export interface CurrenciesResponse {
  fiat: {
    count: number;
    currencies: CurrencyInfo[];
  };
  crypto: {
    count: number;
    currencies: CurrencyInfo[];
  };
  total: number;
}

export interface GameEventData {
  type: 'exit' | 'error' | 'session_expired' | 'balance_low' | 'bet' | 'win';
  data?: Record<string, unknown>;
}

type EventCallback = (event: GameEventData) => void;

const DEFAULT_URLS = {
  production: 'https://api.netspace.one',
  sandbox: 'https://sandbox-api.netspace.one',
};

export class NetspaceSDK {
  private config: Required<NetspaceConfig>;
  private eventListeners: Map<string, EventCallback[]> = new Map();
  private gameIframe: HTMLIFrameElement | null = null;

  constructor(config: NetspaceConfig) {
    const environment = config.environment || 'production';
    
    this.config = {
      apiKey: config.apiKey,
      secretKey: config.secretKey,
      environment,
      baseUrl: config.baseUrl || DEFAULT_URLS[environment],
    };

    // Listen for postMessage events from game iframe
    if (typeof window !== 'undefined') {
      window.addEventListener('message', this.handleGameMessage.bind(this));
    }
  }

  /**
   * Generate HMAC-SHA256 signature for API requests
   */
  private async generateSignature(timestamp: string, body: string): Promise<string> {
    const message = `${timestamp}${body}`;
    
    if (typeof window !== 'undefined' && window.crypto?.subtle) {
      // Browser environment
      const encoder = new TextEncoder();
      const keyData = encoder.encode(this.config.secretKey);
      const messageData = encoder.encode(message);
      
      const key = await window.crypto.subtle.importKey(
        'raw',
        keyData,
        { name: 'HMAC', hash: 'SHA-256' },
        false,
        ['sign']
      );
      
      const signature = await window.crypto.subtle.sign('HMAC', key, messageData);
      return Array.from(new Uint8Array(signature))
        .map(b => b.toString(16).padStart(2, '0'))
        .join('');
    } else {
      // Node.js environment
      const crypto = await import('crypto');
      return crypto.createHmac('sha256', this.config.secretKey)
        .update(message)
        .digest('hex');
    }
  }

  /**
   * Make authenticated API request
   */
  private async request<T>(
    method: 'GET' | 'POST',
    endpoint: string,
    body?: Record<string, unknown>,
    authenticated = true
  ): Promise<T> {
    const url = `${this.config.baseUrl}${endpoint}`;
    const timestamp = Date.now().toString();
    const bodyString = body ? JSON.stringify(body) : '';

    const headers: Record<string, string> = {
      'Content-Type': 'application/json',
    };

    if (authenticated) {
      headers['x-api-key'] = this.config.apiKey;
      headers['x-timestamp'] = timestamp;
      headers['x-signature'] = await this.generateSignature(timestamp, bodyString);
    }

    const response = await fetch(url, {
      method,
      headers,
      body: method === 'POST' ? bodyString : undefined,
    });

    if (!response.ok) {
      const error = await response.json().catch(() => ({ message: 'Unknown error' }));
      throw new NetspaceError(
        error.message || `HTTP ${response.status}`,
        response.status,
        error.code
      );
    }

    return response.json();
  }

  /**
   * Launch a game session (real money)
   */
  async launchGame(params: LaunchGameParams): Promise<LaunchResponse> {
    return this.request<LaunchResponse>('POST', '/game/launch', {
      session_id: params.sessionId,
      player_id: params.playerId,
      game_id: params.gameId,
      currency: params.currency,
      language: params.language,
      return_url: params.returnUrl,
      client_ip: params.clientIp,
      metadata: params.metadata,
    });
  }

  /**
   * Launch a demo game session (no real money)
   */
  async launchDemo(params: LaunchDemoParams): Promise<LaunchResponse> {
    return this.request<LaunchResponse>('POST', '/game/launch-demo', {
      game_id: params.gameId,
      player_id: params.playerId,
      currency: params.currency,
      language: params.language,
      return_url: params.returnUrl,
      client_ip: params.clientIp,
      demo_balance: params.demoBalance,
    }, false);
  }

  /**
   * Get list of supported currencies
   */
  async getCurrencies(): Promise<CurrenciesResponse> {
    return this.request<CurrenciesResponse>('GET', '/game/currencies', undefined, false);
  }

  /**
   * Get game catalog
   */
  async getGames(): Promise<GameInfo[]> {
    return this.request<GameInfo[]>('GET', '/game/catalog', undefined, false);
  }

  /**
   * Open game in an iframe
   */
  openInIframe(containerId: string, gameUrl: string, options?: {
    width?: string;
    height?: string;
    allowFullscreen?: boolean;
  }): HTMLIFrameElement {
    const container = document.getElementById(containerId);
    if (!container) {
      throw new NetspaceError(`Container element '${containerId}' not found`, 404);
    }

    // Remove existing iframe if any
    if (this.gameIframe) {
      this.gameIframe.remove();
    }

    const iframe = document.createElement('iframe');
    iframe.src = gameUrl;
    iframe.style.width = options?.width || '100%';
    iframe.style.height = options?.height || '100%';
    iframe.style.border = 'none';
    iframe.allow = 'autoplay; fullscreen; payment';
    
    if (options?.allowFullscreen !== false) {
      iframe.allowFullscreen = true;
    }

    container.appendChild(iframe);
    this.gameIframe = iframe;

    return iframe;
  }

  /**
   * Open game in a new window/tab
   */
  openInWindow(gameUrl: string, options?: {
    width?: number;
    height?: number;
    name?: string;
  }): Window | null {
    const width = options?.width || 1280;
    const height = options?.height || 720;
    const name = options?.name || 'netspace-game';
    
    const left = (screen.width - width) / 2;
    const top = (screen.height - height) / 2;

    return window.open(
      gameUrl,
      name,
      `width=${width},height=${height},left=${left},top=${top},resizable=yes,scrollbars=no`
    );
  }

  /**
   * Close the game iframe
   */
  closeGame(): void {
    if (this.gameIframe) {
      this.gameIframe.remove();
      this.gameIframe = null;
    }
  }

  /**
   * Subscribe to game events
   */
  on(event: string, callback: EventCallback): void {
    const listeners = this.eventListeners.get(event) || [];
    listeners.push(callback);
    this.eventListeners.set(event, listeners);
  }

  /**
   * Unsubscribe from game events
   */
  off(event: string, callback?: EventCallback): void {
    if (!callback) {
      this.eventListeners.delete(event);
    } else {
      const listeners = this.eventListeners.get(event) || [];
      this.eventListeners.set(event, listeners.filter(cb => cb !== callback));
    }
  }

  /**
   * Handle postMessage events from game iframe
   */
  private handleGameMessage(event: MessageEvent): void {
    // Validate origin
    if (!event.origin.includes('netspace.one')) {
      return;
    }

    const data = event.data as GameEventData;
    if (!data?.type) {
      return;
    }

    // Emit to listeners
    const listeners = this.eventListeners.get(data.type) || [];
    listeners.forEach(callback => callback(data));

    // Also emit to 'all' listeners
    const allListeners = this.eventListeners.get('all') || [];
    allListeners.forEach(callback => callback(data));
  }

  /**
   * Destroy SDK instance and cleanup
   */
  destroy(): void {
    this.closeGame();
    this.eventListeners.clear();
    
    if (typeof window !== 'undefined') {
      window.removeEventListener('message', this.handleGameMessage.bind(this));
    }
  }
}

/**
 * Custom error class for Netspace SDK
 */
export class NetspaceError extends Error {
  public statusCode: number;
  public code?: string;

  constructor(message: string, statusCode: number, code?: string) {
    super(message);
    this.name = 'NetspaceError';
    this.statusCode = statusCode;
    this.code = code;
  }
}

// Default export for convenience
export default NetspaceSDK;
