import hashlib
import hmac
import json
import time
from typing import Any, Dict, List, Optional

import requests

from .exceptions import NetspaceException


class NetspaceSDK:
    """Python SDK for Netspace ONE game integration."""

    def __init__(
        self,
        api_key: str,
        secret_key: str,
        environment: str = 'production',
        base_url: Optional[str] = None,
        timeout: int = 30
    ):
        """
        Initialize the Netspace SDK.

        Args:
            api_key: Your operator API key
            secret_key: Your operator secret key
            environment: 'production' or 'sandbox'
            base_url: Custom API base URL (optional)
            timeout: Request timeout in seconds
        """
        if not api_key or not secret_key:
            raise NetspaceException('API Key and Secret Key are required')

        self.api_key = api_key
        self.secret_key = secret_key
        self.timeout = timeout

        if base_url:
            self.base_url = base_url
        elif environment == 'sandbox':
            self.base_url = 'https://sandbox-api.netspace.one'
        else:
            self.base_url = 'https://api.netspace.one'

    def launch_game(
        self,
        session_id: str,
        player_id: str,
        game_id: str,
        currency: str,
        client_ip: str,
        language: str = 'en',
        return_url: Optional[str] = None,
        metadata: Optional[Dict[str, Any]] = None
    ) -> Dict[str, Any]:
        """
        Launch a real-money game session.

        Args:
            session_id: Your internal session ID
            player_id: Your internal player ID
            game_id: Game identifier (e.g., 'lotterix')
            currency: Player's currency code
            client_ip: Player's IP address
            language: UI language (ISO 639-1)
            return_url: URL to redirect on exit
            metadata: Optional custom metadata

        Returns:
            Dict with game_url, session_token, expires_at, etc.
        """
        return self._request('POST', '/game/launch', {
            'session_id': session_id,
            'player_id': player_id,
            'game_id': game_id,
            'currency': currency,
            'language': language,
            'return_url': return_url,
            'client_ip': client_ip,
            'metadata': metadata,
        })

    def launch_demo(
        self,
        game_id: str,
        currency: str = 'USD',
        language: str = 'en',
        return_url: Optional[str] = None,
        demo_balance: int = 10000
    ) -> Dict[str, Any]:
        """
        Launch a demo game session.

        Args:
            game_id: Game identifier
            currency: Currency for display
            language: UI language
            return_url: URL to redirect on exit
            demo_balance: Demo balance in cents

        Returns:
            Dict with game_url, session_token, etc.
        """
        return self._request('POST', '/game/launch-demo', {
            'game_id': game_id,
            'currency': currency,
            'language': language,
            'return_url': return_url,
            'demo_balance': demo_balance,
        })

    def get_currencies(self) -> Dict[str, Any]:
        """
        Get list of supported currencies.

        Returns:
            Dict with fiat and crypto currency lists
        """
        return self._request('GET', '/game/currencies')

    def get_games(self) -> List[Dict[str, Any]]:
        """
        Get available games catalog.

        Returns:
            List of game objects
        """
        return self._request('GET', '/game/games')

    def _request(
        self,
        method: str,
        path: str,
        body: Optional[Dict[str, Any]] = None
    ) -> Any:
        """Make an authenticated API request."""
        url = f"{self.base_url}{path}"
        timestamp = str(int(time.time() * 1000))
        body_json = json.dumps(body) if body else ''

        signature = self._sign(timestamp, body_json)

        headers = {
            'Content-Type': 'application/json',
            'X-API-Key': self.api_key,
            'X-Timestamp': timestamp,
            'X-Signature': signature,
        }

        try:
            if method == 'GET':
                response = requests.get(url, headers=headers, timeout=self.timeout)
            else:
                response = requests.post(
                    url,
                    headers=headers,
                    data=body_json,
                    timeout=self.timeout
                )

            data = response.json()

            if response.status_code >= 400:
                message = data.get('message', 'Unknown error')
                error_code = data.get('code', 'UNKNOWN_ERROR')
                raise NetspaceException(message, response.status_code, error_code)

            return data

        except requests.RequestException as e:
            raise NetspaceException(f'HTTP request failed: {str(e)}')

    def _sign(self, timestamp: str, body: str) -> str:
        """Generate HMAC-SHA256 signature."""
        string_to_sign = timestamp + body
        return hmac.new(
            self.secret_key.encode('utf-8'),
            string_to_sign.encode('utf-8'),
            hashlib.sha256
        ).hexdigest()
