# Netspace ONE PHP SDK

PHP SDK for integrating Netspace ONE games into your platform.

## Requirements

- PHP 7.4 or higher
- cURL extension
- JSON extension

## Installation

### Composer

```bash
composer require netspace/sdk
```

### Manual

Download the SDK and include the autoloader:

```php
require_once 'path/to/sdk-php/src/NetspaceSDK.php';
require_once 'path/to/sdk-php/src/NetspaceException.php';
```

## Quick Start

```php
<?php

use Netspace\NetspaceSDK;
use Netspace\NetspaceException;

// Initialize SDK
$netspace = new NetspaceSDK([
    'apiKey' => 'your-api-key',
    'secretKey' => 'your-secret-key',
    'environment' => 'production' // or 'sandbox'
]);

// Launch a game
try {
    $session = $netspace->launchGame([
        'sessionId' => 'your-session-id',
        'playerId' => 'player-123',
        'gameId' => 'lotterix',
        'currency' => 'USD',
        'language' => 'en',
        'returnUrl' => 'https://your-site.com/lobby',
        'clientIp' => $_SERVER['REMOTE_ADDR']
    ]);

    // Redirect player to game
    header('Location: ' . $session['game_url']);
    exit;

} catch (NetspaceException $e) {
    echo "Error: " . $e->getMessage();
    echo "Code: " . $e->getErrorCode();
}
```

## API Reference

### Constructor

```php
$netspace = new NetspaceSDK([
    'apiKey' => 'your-api-key',      // Required
    'secretKey' => 'your-secret-key', // Required
    'environment' => 'production',    // Optional: 'production' or 'sandbox'
    'baseUrl' => 'https://...',       // Optional: custom API URL
    'timeout' => 30                   // Optional: request timeout in seconds
]);
```

### Methods

#### `launchGame(array $params): array`

Launch a real-money game session.

```php
$session = $netspace->launchGame([
    'sessionId' => 'unique-session-id',  // Required
    'playerId' => 'player-123',          // Required
    'gameId' => 'lotterix',              // Required
    'currency' => 'USD',                 // Required
    'language' => 'en',                  // Optional
    'returnUrl' => 'https://...',        // Optional
    'clientIp' => '192.168.1.1',         // Required
    'metadata' => ['custom' => 'data']   // Optional
]);
```

**Response:**
```php
[
    'operator_session_id' => 'your-session-id',
    'netspace_session_id' => 'ns-internal-id',
    'session_token' => 'jwt-token',
    'game_url' => 'https://games.netspace.one/lotterix?token=...',
    'expires_at' => '2025-01-15T12:00:00.000Z'
]
```

#### `launchDemo(array $params): array`

Launch a demo game session.

```php
$session = $netspace->launchDemo([
    'gameId' => 'lotterix',      // Required
    'currency' => 'USD',         // Optional
    'language' => 'en',          // Optional
    'demoBalance' => 10000       // Optional: balance in cents
]);
```

#### `getCurrencies(): array`

Get list of supported currencies.

```php
$currencies = $netspace->getCurrencies();
// ['fiat' => ['count' => 18, 'currencies' => [...]], 'crypto' => [...]]
```

#### `getGames(): array`

Get available games catalog.

```php
$games = $netspace->getGames();
// [['id' => 'lotterix', 'name' => 'Lotterix', 'rtp' => 72, ...]]
```

## Error Handling

```php
use Netspace\NetspaceException;

try {
    $session = $netspace->launchGame([...]);
} catch (NetspaceException $e) {
    $message = $e->getMessage();        // Error message
    $statusCode = $e->getStatusCode();  // HTTP status code
    $errorCode = $e->getErrorCode();    // Error code (e.g., 'INSUFFICIENT_BALANCE')
}
```

## Laravel Integration

```php
// config/services.php
'netspace' => [
    'api_key' => env('NETSPACE_API_KEY'),
    'secret_key' => env('NETSPACE_SECRET_KEY'),
    'environment' => env('NETSPACE_ENV', 'production'),
],

// app/Providers/AppServiceProvider.php
use Netspace\NetspaceSDK;

public function register()
{
    $this->app->singleton(NetspaceSDK::class, function ($app) {
        return new NetspaceSDK([
            'apiKey' => config('services.netspace.api_key'),
            'secretKey' => config('services.netspace.secret_key'),
            'environment' => config('services.netspace.environment'),
        ]);
    });
}

// In your controller
public function launchGame(Request $request, NetspaceSDK $netspace)
{
    $session = $netspace->launchGame([
        'sessionId' => $request->session()->getId(),
        'playerId' => auth()->id(),
        'gameId' => 'lotterix',
        'currency' => 'USD',
        'clientIp' => $request->ip()
    ]);

    return redirect($session['game_url']);
}
```

## WordPress Integration

```php
// In your plugin or theme
function netspace_launch_game() {
    require_once 'path/to/sdk-php/src/NetspaceSDK.php';
    
    $netspace = new \Netspace\NetspaceSDK([
        'apiKey' => get_option('netspace_api_key'),
        'secretKey' => get_option('netspace_secret_key'),
    ]);

    $user = wp_get_current_user();
    
    $session = $netspace->launchGame([
        'sessionId' => wp_generate_uuid4(),
        'playerId' => (string) $user->ID,
        'gameId' => 'lotterix',
        'currency' => 'USD',
        'clientIp' => $_SERVER['REMOTE_ADDR']
    ]);

    wp_redirect($session['game_url']);
    exit;
}
```

## Support

- Documentation: https://netspace.one/api
- Email: tech@netspace.one

## License

MIT
