# Netspace ONE Python SDK

Python SDK for integrating Netspace ONE games into your platform.

## Requirements

- Python 3.7 or higher
- requests library

## Installation

### pip

```bash
pip install netspace-sdk
```

### Manual

```bash
git clone https://github.com/netspace-one/sdk-python.git
cd sdk-python
pip install -e .
```

## Quick Start

```python
from netspace import NetspaceSDK, NetspaceException

# Initialize SDK
netspace = NetspaceSDK(
    api_key='your-api-key',
    secret_key='your-secret-key',
    environment='production'  # or 'sandbox'
)

# Launch a game
try:
    session = netspace.launch_game(
        session_id='your-session-id',
        player_id='player-123',
        game_id='lotterix',
        currency='USD',
        language='en',
        return_url='https://your-site.com/lobby',
        client_ip='192.168.1.1'
    )

    # Redirect player to game
    game_url = session['game_url']
    print(f"Game URL: {game_url}")

except NetspaceException as e:
    print(f"Error: {e.message}")
    print(f"Code: {e.error_code}")
```

## API Reference

### Constructor

```python
netspace = NetspaceSDK(
    api_key='your-api-key',       # Required
    secret_key='your-secret-key',  # Required
    environment='production',      # Optional: 'production' or 'sandbox'
    base_url='https://...',        # Optional: custom API URL
    timeout=30                     # Optional: request timeout in seconds
)
```

### Methods

#### `launch_game()`

Launch a real-money game session.

```python
session = netspace.launch_game(
    session_id='unique-session-id',  # Required
    player_id='player-123',          # Required
    game_id='lotterix',              # Required
    currency='USD',                  # Required
    client_ip='192.168.1.1',         # Required
    language='en',                   # Optional
    return_url='https://...',        # Optional
    metadata={'custom': 'data'}      # Optional
)
```

**Response:**
```python
{
    'operator_session_id': 'your-session-id',
    'netspace_session_id': 'ns-internal-id',
    'session_token': 'jwt-token',
    'game_url': 'https://games.netspace.one/lotterix?token=...',
    'expires_at': '2025-01-15T12:00:00.000Z'
}
```

#### `launch_demo()`

Launch a demo game session.

```python
session = netspace.launch_demo(
    game_id='lotterix',      # Required
    currency='USD',          # Optional
    language='en',           # Optional
    demo_balance=10000       # Optional: balance in cents
)
```

#### `get_currencies()`

Get list of supported currencies.

```python
currencies = netspace.get_currencies()
# {'fiat': {'count': 18, 'currencies': [...]}, 'crypto': {...}}
```

#### `get_games()`

Get available games catalog.

```python
games = netspace.get_games()
# [{'id': 'lotterix', 'name': 'Lotterix', 'rtp': 72, ...}]
```

## Error Handling

```python
from netspace import NetspaceSDK, NetspaceException

try:
    session = netspace.launch_game(...)
except NetspaceException as e:
    print(e.message)       # Error message
    print(e.status_code)   # HTTP status code
    print(e.error_code)    # Error code (e.g., 'INSUFFICIENT_BALANCE')
```

## Django Integration

```python
# settings.py
NETSPACE_API_KEY = os.environ.get('NETSPACE_API_KEY')
NETSPACE_SECRET_KEY = os.environ.get('NETSPACE_SECRET_KEY')
NETSPACE_ENV = os.environ.get('NETSPACE_ENV', 'production')

# views.py
from django.shortcuts import redirect
from django.conf import settings
from netspace import NetspaceSDK

def launch_game(request):
    netspace = NetspaceSDK(
        api_key=settings.NETSPACE_API_KEY,
        secret_key=settings.NETSPACE_SECRET_KEY,
        environment=settings.NETSPACE_ENV
    )

    session = netspace.launch_game(
        session_id=request.session.session_key,
        player_id=str(request.user.id),
        game_id='lotterix',
        currency='USD',
        client_ip=request.META.get('REMOTE_ADDR')
    )

    return redirect(session['game_url'])
```

## FastAPI Integration

```python
from fastapi import FastAPI, Request
from fastapi.responses import RedirectResponse
from netspace import NetspaceSDK
import os

app = FastAPI()

netspace = NetspaceSDK(
    api_key=os.environ['NETSPACE_API_KEY'],
    secret_key=os.environ['NETSPACE_SECRET_KEY']
)

@app.get("/launch/{game_id}")
async def launch_game(game_id: str, request: Request):
    session = netspace.launch_game(
        session_id=f"session-{request.state.user_id}",
        player_id=str(request.state.user_id),
        game_id=game_id,
        currency='USD',
        client_ip=request.client.host
    )
    return RedirectResponse(session['game_url'])
```

## Flask Integration

```python
from flask import Flask, redirect, request, session
from netspace import NetspaceSDK
import os

app = Flask(__name__)

netspace = NetspaceSDK(
    api_key=os.environ['NETSPACE_API_KEY'],
    secret_key=os.environ['NETSPACE_SECRET_KEY']
)

@app.route('/launch/<game_id>')
def launch_game(game_id):
    game_session = netspace.launch_game(
        session_id=session.sid,
        player_id=str(session['user_id']),
        game_id=game_id,
        currency='USD',
        client_ip=request.remote_addr
    )
    return redirect(game_session['game_url'])
```

## Support

- Documentation: https://netspace.one/api
- Email: tech@netspace.one

## License

MIT
